﻿using Santec.Hardware.Devices.Power;
using Santec.Hardware.Services;
using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;

namespace OPMSample
{
    /// <summary>
    /// This program provides sample code of operating an OPM via the Santec Hardware library. It should be noted that it does not cover the full range of OPM capabilities.
    /// </summary>
    class Program
    {
        static async Task Main(string[] args)
        {
            List<IOPM> detectedOPMs;
            IOPM opm;

            double reading;

            List<int> readingWavelengths = new List<int>() { 1310, 1550 };

            IHardwareDetectionService hardwareDetectionService = new HardwareDetectionService();

            ConsoleKeyInfo keyInfo;

            List<Task<IReadOnlyList<double>>> loggingTasks = new List<Task<IReadOnlyList<double>>>();

            // Finding the OPM
            Console.WriteLine("Finding USB OPM...");

            // Step 1. Find all the USB OPMs
            detectedOPMs = await hardwareDetectionService.DetectHardwareAsync<IOPM>(EConnectionTypesToDetect.USB);

            if (!detectedOPMs.Any())
            {
                throw new Exception("Could not find OPM.");
            }

            // Step 2. Select the first OPM to use for the sample
            opm = detectedOPMs.First();

            Console.WriteLine("OPM Found. Initializing...");

            // Step 3. Initialize the OPM
            await opm.InitializeAsync();

            Console.WriteLine("OPM initialized.");

            // Setting reading properties
            Console.WriteLine("Setting OPM reading settings...");

            // Step 1. Set the reading resolution to 3 digits for all detectors.
            // Options are 1, 2, or 3 digits.
            await opm.SetAllDetectorResolutionsAsync(EResolution.ThreeDigits);

            // Step 2. Set the averaging time to 20ms for all detectors.
            // The averaging time must be between 0.05ms and 1000ms
            await opm.SetAllDetectorAveragingTimesAsync(20);

            //Step 3. Set the OPM to absolute power for all detectors.
            await opm.SetAllDetectorReadingModesAsync(EReadingMode.Power);

            //Step 4. Set the OPM to use automatic gain ranges for all detectors.
            for(int i = 0; i < opm.NumberOfDetectors; i++)
            {
                await opm.SetDetectorGainRangeAsync(EGainRange.Auto);
            }

            Console.WriteLine("Done setting OPM reading settings.");

            // Use the OPM to take measurement readings at 1310nm and 1550nm
            Console.WriteLine("Taking readings using the OPM...");

            foreach(int wavelength in readingWavelengths)
            {
                // Step 1. Set the wavelength for all detectors
                // The wavelength must be between 400nm and 1700nm
                Console.WriteLine($"Setting all OPM detectors to {wavelength}nm...");

                await opm.SetAllDetectorWavelengthsAsync(wavelength);

                Console.WriteLine($"All OPM detectors set to {wavelength}nm.");

                // Step 2. Take readings for all detectors
                // Note: All detectors can be read together with "await opm.ReadAllDetectorsAsync(wavelength);"
                Console.WriteLine($"Reading all OPM detectors at {wavelength}nm.");

                for (int i = 0; i < opm.NumberOfDetectors; i++)
                {
                    Console.WriteLine($"Reading detector {i} at {wavelength}nm...");

                    reading = await opm.ReadDetectorAsync(i, wavelength);

                    Console.WriteLine($"Detector {i} power at {wavelength}: {reading}");
                }
            }

            // Use the OPM to log all detectors
            Console.WriteLine("Log Detectors? Y for yes. Any other key to exit...");

            keyInfo = Console.ReadKey();

            if (keyInfo.Key == ConsoleKey.Y)
            {
                Console.WriteLine();

                // Step 1. Set the wavelength for all detectors
                Console.WriteLine($"Setting all OPM detectors to 1310nm...");

                await opm.SetAllDetectorWavelengthsAsync(1310);

                Console.WriteLine($"All OPM detectors set to 1310nm.");

                // Step 2. Start the logging for each detector
                for (int i = 0; i < opm.NumberOfDetectors; i++)
                {
                    // The number of logging points must be between 1 and 128000
                    // We are logging 1000 points at 20MS using free run mode
                    loggingTasks.Add(opm.LogDetectorAsync(i, 1000, 20, ETriggerEdge.Falling, ETriggerBehaviourMode.FreeRun));

                    Console.WriteLine($"Logging started for detector {i}...");
                }

                // Step 3. Wait for the logging to complete.
                await Task.WhenAll(loggingTasks);

                Console.WriteLine($"Logging finished for all detectors.");

                // Show the results for each detector
                for (int i = 0; i < opm.NumberOfDetectors; i++)
                {
                    Console.WriteLine($"Logging data for detector {i}:");

                    Console.WriteLine(string.Join(", ", loggingTasks[i].Result));
                }

                Console.WriteLine("Press any key to exit...");

                Console.ReadKey();
            }
        }
    }
}
